<?php

namespace App\Http\Controllers;

use App\Models\Certification;
use App\Models\CandidateProfile;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Mpdf\Mpdf;

class CertificationController extends Controller
{
    public function index(): JsonResponse
    {
        $certifications = Certification::with(['user.candidateProfile', 'course'])->get();
        return response()->json([
            'status' => 'success',
            'data' => $certifications,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $certification = Certification::with(['user.candidateProfile', 'course'])->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $certification,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'User_id' => 'required|exists:Users,User_id',
            'Course_id' => 'required|exists:Courses,Course_id',
            'Certificate_url' => 'nullable|string|max:255',
            'Issue_date' => 'required|date',
            'Expiry_date' => 'nullable|date|after_or_equal:Issue_date',
            'Verification_code' => 'nullable|string|max:50|unique:Certifications,Verification_code',
            'Serial_number' => 'nullable|string|max:50|unique:Certifications,Serial_number',
            'Status' => 'sometimes|in:issued,revoked,pending',
            'Certificate_version' => 'nullable|string|max:20',
            'LinkedIn_shared' => 'sometimes|boolean', // Added
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $certification = Certification::create(array_merge(
            $request->only([
                'User_id',
                'Course_id',
                'Certificate_url',
                'Issue_date',
                'Expiry_date',
                'Verification_code',
                'Serial_number',
                'Status',
                'Certificate_version',
                'LinkedIn_shared', // Added
            ]),
            ['Created_at' => now()]
        ));

        // Load user.candidateProfile and course relationships
        $certification->load(['user.candidateProfile', 'course']);

        try {
            $this->generateCertificatePdf($certification);
        } catch (\Exception $e) {
            Log::error('Failed to generate certificate PDF: ' . $e->getMessage(), ['certification_id' => $certification->Certification_id]);
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to generate certificate PDF: ' . $e->getMessage(),
            ], 500);
        }

        return response()->json([
            'status' => 'success',
            'data' => $certification,
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $certification = Certification::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'User_id' => 'sometimes|exists:Users,User_id',
            'Course_id' => 'sometimes|exists:Courses,Course_id',
            'Certificate_url' => 'nullable|string|max:255',
            'Issue_date' => 'sometimes|date',
            'Expiry_date' => 'nullable|date|after_or_equal:Issue_date',
            'Verification_code' => 'nullable|string|max:50|unique:Certifications,Verification_code,' . $certification->Certification_id,
            'Serial_number' => 'nullable|string|max:50|unique:Certifications,Serial_number,' . $certification->Certification_id,
            'Status' => 'sometimes|in:issued,revoked,pending',
            'Certificate_version' => 'nullable|string|max:20',
            'LinkedIn_shared' => 'sometimes|boolean', // Added
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $certification->update($request->only([
            'User_id',
            'Course_id',
            'Certificate_url',
            'Issue_date',
            'Expiry_date',
            'Verification_code',
            'Serial_number',
            'Status',
            'Certificate_version',
            'LinkedIn_shared', // Added
        ]));

        // Load user.candidateProfile and course relationships
        $certification->load(['user.candidateProfile', 'course']);

        try {
            $this->generateCertificatePdf($certification);
        } catch (\Exception $e) {
            Log::error('Failed to regenerate certificate PDF: ' . $e->getMessage(), ['certification_id' => $certification->Certification_id]);
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to regenerate certificate PDF: ' . $e->getMessage(),
            ], 500);
        }

        return response()->json([
            'status' => 'success',
            'data' => $certification,
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $certification = Certification::findOrFail($id);
        if ($certification->Certificate_url) {
            Storage::disk('public')->delete(str_replace('/storage/', '', $certification->Certificate_url));
        }
        $certification->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Certification deleted successfully',
        ], 200);
    }

    public function download($id)
    {
        $certification = Certification::findOrFail($id);
        $filePath = public_path($certification->Certificate_url);

        if (file_exists($filePath)) {
            return response()->download($filePath, 'certificate_' . $id . '.pdf');
        }

        return response()->json([
            'status' => 'error',
            'message' => 'Certificate file not found',
        ], 404);
    }


}
